<?php

namespace App\Http\Controllers;

use App\Models\Subscription;
use App\Models\Plan;
use App\Models\CompanyProfile;
use App\Models\SubscriptionHistory;
use App\Http\Controllers\PaymentController;
use App\Models\Transaction;
use App\Models\Payment;
use App\Services\PaymentHandler;
use App\Models\CompanyFeatureUsage;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class SubscriptionController extends Controller
{

    public function index(): JsonResponse
    {
        $subscriptions = Subscription::all();
        return response()->json($subscriptions);
    }

    public function show($id): JsonResponse
    {
        $subscription = Subscription::find($id);
        if (!$subscription) {
            return response()->json(['message' => 'Subscription not found'], 404);
        }
        return response()->json($subscription);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Company_id' => 'required|integer|exists:company_profiles,Company_id',
            'Plan_id' => 'required|integer|exists:plans,Plan_id',
            'Amount' => 'required|numeric|min:1',
            'Payment_gateway' => 'required|in:razorpay',
            'Currency' => 'required|string',
            'Start_date' => 'required|date',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $companyId = $request->input('Company_id');
        $planId = $request->input('Plan_id');

        // Temporary order ID for new subscription
        $orderId = 'new_order_' . $companyId . '_' . $planId . '_' . time();

        return response()->json([
            'status' => 'pending',
            'message' => 'Payment initiation successful. Please complete the payment.',
            'data' => [
                'key' => env('RAZORPAY_KEY', 'your_razorpay_key'),
                'amount' => $request->input('Amount') * 100,
                'currency' => $request->input('Currency'),
                'name' => 'New Subscription for Company ' . $companyId,
                'description' => 'New subscription payment',
                'order_id' => $orderId,
            ],
        ], 200);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $subscription = Subscription::find($id);
        if (!$subscription) {
            return response()->json(['message' => 'Subscription not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'Company_id' => 'sometimes|integer|exists:company_profiles,Company_id',
            'Plan_id' => 'sometimes|integer|exists:plans,Plan_id',
            'Start_date' => 'sometimes|date',
            'End_date' => 'sometimes|date|after:Start_date',
            'Status' => 'sometimes|string|in:active,inactive,expired',
            'payment_method' => 'sometimes|string|in:paypal,stripe,razorpay',
            'renewal_count' => 'sometimes|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $subscription->update($request->all());
        return response()->json(['message' => 'Subscription updated successfully', 'data' => $subscription], 200);
    }

    public function destroy($id): JsonResponse
    {
        $subscription = Subscription::find($id);
        if (!$subscription) {
            return response()->json(['message' => 'Subscription not found'], 404);
        }
        $subscription->delete();
        return response()->json(['message' => 'Subscription deleted successfully'], 200);
    }

    public function upgrade(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Company_id' => 'required|integer|exists:company_profiles,Company_id',
            'New_Plan_id' => 'required|integer|exists:plans,Plan_id',
            'Amount' => 'required|numeric|min:1',
            'Payment_gateway' => 'required|in:razorpay',
            'Subscription_id' => 'nullable|integer', // Optional for upgrades
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $companyId = $request->input('Company_id');
        $newPlanId = $request->input('New_Plan_id');
        $subscriptionId = $request->input('Subscription_id');
        $amount = $request->input('Amount');
        $currency = $request->input('Currency', 'USD');
        $paymentGateway = $request->input('Payment_gateway');

        // Create a temporary payment record
        $payment = Payment::create([
            'Payment_status' => 'pending',
            'Payment_method' => $paymentGateway,
            'Amount' => $amount,
            'Currency' => $currency,
        ]);

        try {
            $paymentHandler = new PaymentHandler($paymentGateway);
            $response = $paymentHandler->initiatePayment($payment);

            $orderId = $response['transaction_id'] ?? 'upgrade_order_' . ($subscriptionId ?: $companyId) . '_' . time();

            return response()->json([
                'status' => 'pending',
                'message' => 'Payment initiation successful. Please complete the payment.',
                'data' => [
                    'key' => env('RAZORPAY_KEY', 'rzp_test_4gloqgoog9Iz18'),
                    'amount' => $amount * 100,
                    'currency' => $currency,
                    'name' => $subscriptionId ? 'Upgrade Subscription for Company ' . $companyId : 'New Subscription for Company ' . $companyId,
                    'description' => $subscriptionId ? 'Upgrade subscription payment' : 'New subscription payment',
                    'order_id' => $orderId,
                ],
            ], 200);
        } catch (\Exception $e) {
            Log::error('Payment initiation failed: ' . $e->getMessage());
            $payment->update(['Payment_status' => 'failed']);
            return response()->json(['message' => 'Payment initiation failed: ' . $e->getMessage()], 400);
        }
    }

    public function callback(Request $request, $gateway)
    {
        $validator = Validator::make($request->all(), [
            'Transaction_id' => 'required|string',
            'razorpay_payment_id' => 'required',
            'razorpay_order_id' => 'required',
            'razorpay_signature' => 'required',
            'Company_id' => 'required|integer|exists:company_profiles,Company_id',
            'New_Plan_id' => 'sometimes|required|integer|exists:plans,Plan_id',
            'Subscription_id' => 'sometimes|integer|exists:subscriptions,Subscription_id',
            'Amount' => 'required|numeric|min:1',
            'Currency' => 'required|string',
        ]);

        if ($validator->fails()) {
            Log::error('Callback validation failed: ' . json_encode($validator->errors()));
            return response()->json($validator->errors(), 422);
        }

        $companyId = $request->input('Company_id');
        $planId = $request->input('New_Plan_id'); // Use New_Plan_id for upgrades
        $subscriptionId = $request->input('Subscription_id');
        $orderId = $request->input('razorpay_order_id');
        $amount = $request->input('Amount');
        $currency = $request->input('Currency');
        $now = Carbon::now();
        $plan = Plan::find($planId);

        try {
            $paymentHandler = new PaymentHandler('razorpay');
            $gatewayData = [
                'razorpay_payment_id' => $request->input('razorpay_payment_id'),
                'razorpay_signature' => $request->input('razorpay_signature'),
            ];
            $paymentHandler->verifyPayment($orderId, $gatewayData);

            // Create Transaction
            $transaction = Transaction::create([
                'Amount' => $amount,
                'Currency' => $currency,
                'Transaction_status' => 'completed',
                'Payment_gateway' => 'razorpay',
                'Transaction_id' => $orderId,
            ]);

            // Update or create Payment record
            $payment = Payment::where('Transaction_id', $orderId)->first();
            if ($payment) {
                $payment->update(['Payment_status' => 'completed', 'razorpay_payment_id' => $request->input('razorpay_payment_id')]);
            } else {
                $payment = Payment::create([
                    'Transaction_id' => $transaction->Transaction_id,
                    'Payment_status' => 'completed',
                    'Payment_method' => 'razorpay',
                    'razorpay_payment_id' => $request->input('razorpay_payment_id'),
                ]);
            }

            $remainingDays = 0;
            if ($subscriptionId) {
                $currentSubscription = Subscription::find($subscriptionId);
                if ($currentSubscription) {
                    $remainingDays = $currentSubscription->End_date ? Carbon::now()->diffInDays($currentSubscription->End_date, false) : 0;
                    $remainingDays = max(0, $remainingDays);
                    $currentSubscription->update([
                        'Plan_id' => $planId,
                        'Start_date' => $now,
                        'End_date' => $now->copy()->addDays($remainingDays + $plan->Duration + ($plan->trial_period ?? 0)),
                        'Status' => 'active',
                        'renewal_count' => ($currentSubscription->renewal_count ?? 0) + 1,
                    ]);
                    $subscription = $currentSubscription;
                } else {
                    Log::error('Invalid Subscription_id: ' . $subscriptionId);
                    return response()->json(['Subscription_id' => ['The selected subscription id is invalid.']], 422);
                }
            } else {
                $subscription = Subscription::create([
                    'Company_id' => $companyId,
                    'Plan_id' => $planId,
                    'Start_date' => $now,
                    'End_date' => $now->copy()->addDays($plan->Duration + ($plan->trial_period ?? 0)),
                    'Status' => 'active',
                    'Amount' => $amount,
                    'Currency' => $currency,
                    'Payment_gateway' => 'razorpay',
                    'renewal_count' => 0,
                ]);
            }

            $transaction->update(['Subscription_id' => $subscription->Subscription_id]);

            Log::info('Payment successful for Company_id: ' . $companyId . ', Subscription_id: ' . $subscription->Subscription_id);
            return response()->json([
                'status' => 'success',
                'message' => 'Payment successful. Subscription activated.',
                'data' => $subscription,
            ], 200);
        } catch (\Exception $e) {
            Log::error('Payment verification failed: ' . $e->getMessage());
            return response()->json(['message' => 'Payment verification failed: ' . $e->getMessage()], 400);
        }
    }

    public function showUpgradeForm($companyId)
    {
        $plans = Plan::with('currency')->get();
        $companyProfile = CompanyProfile::find($companyId);

        if (!$companyProfile) {
            Log::error('Company not found for companyId: ' . $companyId);
            return redirect()->back()->with('error', 'Company not found.');
        }

        $subscription = Subscription::where('Company_id', $companyId)
            ->where('Status', 'active')
            ->first();

        return view('upgrade-plan', compact('plans', 'subscription', 'companyProfile'));
    }

    public function getSubscriptionByName(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $subscriptions = Subscription::where('Name', $request->name)->get();

        if ($subscriptions->isEmpty()) {
            return response()->json(['message' => 'No subscriptions found with this name'], 404);
        }

        return response()->json($subscriptions);
    }

    public function getSubscriptionHistory($company_id): JsonResponse
    {
        $validator = Validator::make(['company_id' => $company_id], [
            'company_id' => 'required|integer|exists:company_profiles,Company_id',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $history = SubscriptionHistory::where('company_id', $company_id)
            ->with('plan')
            ->orderBy('start_date', 'desc')
            ->get();

        if ($history->isEmpty()) {
            return response()->json(['message' => 'No subscription history found'], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Subscription history retrieved successfully',
            'data' => $history
        ], 200);
    }

    public function checkAndDeactivateSubscriptions()
    {
        $subscriptions = Subscription::where('Status', 'active')->get();
        foreach ($subscriptions as $subscription) {
            if ($subscription->End_date && Carbon::parse($subscription->End_date)->lte(now())) {
                $subscription->Status = 'inactive';
                $subscription->save();

                SubscriptionHistory::where('subscription_id', $subscription->Subscription_id)
                    ->where('status', 'active')
                    ->update([
                        'end_date' => $subscription->End_date->toDateTimeString(),
                        'status' => 'inactive',
                    ]);

                CompanyFeatureUsage::where('Company_id', $subscription->Company_id)
                    ->where('Subscription_id', $subscription->Subscription_id)
                    ->whereNull('End_Date')
                    ->update(['End_Date' => $subscription->End_date]);
            }
        }
    }
}
